/**
  ******************************************************************************
  * @file    fibonacci.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   Helper functions
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

#include "fibonacci.h"

extern volatile unsigned long g_MSCtr;

/** @brief Fully recursive implementation of the Fibonacci function
 *
 *
 *  @param p_Arg Argument to the function
 *  @return Fibonacci(p_Arg)
 */

unsigned long Fibonacci_Recursive(unsigned long p_Arg)
{
    switch(p_Arg)
    {
        case 0: 
        case 1: return p_Arg;
        default: return Fibonacci_Recursive(p_Arg-1)+Fibonacci_Recursive(p_Arg-2);
    };
}

/** @brief Fully tail recursive implementation of the Fibonacci function
 *
 *  @param p_Arg Argument to the function
 *  @param p_Acc1 First accumulative parameter
 *  @param p_Acc2 Second accumulative parameter
 *  @return Fibonacci(p_Arg) (top level)
 */

unsigned long FiboTail(unsigned long p_Arg, unsigned long p_Acc1, unsigned long p_Acc2)
{
    switch(p_Arg)
    {
        case 0: return p_Acc1;
        case 1: return p_Acc2;
        default: return FiboTail(p_Arg - 1,p_Acc2,p_Acc2+p_Acc1); 
    }
}

/** @brief Partially tail recursive implementation of the Fibonacci function
 *
 *  @param p_Arg Argument to the function
 *  @param p_Accum accumulative parameter
 *  @return Fibonacci(p_Arg) (top level)
 */


unsigned long Fibonacci_Unwind(unsigned long p_Arg,unsigned long p_Accum)
{
    if (p_Arg < 2)
        return (p_Arg + p_Accum);
    else
        return (Fibonacci_Unwind(p_Arg - 1, Fibonacci_Unwind(p_Arg - 2,p_Accum)));
}

/** @brief Wrapper around the Fibonacci function
 *
 *
 *  @param p_Arg Argument to the function
 *  @param p_TimeNeeded Return parameter: receives the numer of milliseconds elapsed between beginning and end of computation
 *  @param p_CyclesNeeded Return parameter: receives the number of CPU cycles elapsed between beginning and end of computation. Only reliable when *p_TimeNeeded < wrap around! 
 *  @param p_Implementation member of the enumerator FIBTEST_USE_XXX
 *  @return Fibonacci(p_Arg)
 */

unsigned long Fibonacci(unsigned long p_Arg,unsigned long *p_TimeNeeded,unsigned long *p_CyclesNeeded,unsigned long p_Implementation)
{
    unsigned long a_Res;
    volatile unsigned long a_TimeBefore = g_MSCtr;
    (*(volatile unsigned int*)0xE0001004) = 0;   // reset dwt cycle counter 
    switch (p_Implementation)
    {
        case FIBTEST_USE_FULLY_RECURSIVE_IMPLEMENTATION: a_Res = Fibonacci_Recursive(p_Arg); break;
        case FIBTEST_USE_FULLY_TAIL_RECURSIVE_IMPLEMENTATION: a_Res = Fibonacci_Recursive(p_Arg); break;
        case FIBTEST_USE_PARTIALLY_TAIL_RECURSIVE_IMPLEMENTATION: a_Res = Fibonacci_Unwind(p_Arg,0); break;
    }  
    volatile unsigned long a_TimeAfter = g_MSCtr;
    *p_CyclesNeeded = (*(volatile unsigned int*)0xE0001004);    
    *p_TimeNeeded = a_TimeAfter - a_TimeBefore;
    return a_Res;
}

